package info.u_team.u_team_core.util;

import org.joml.Matrix4f;

import com.mojang.blaze3d.systems.RenderSystem;
import net.minecraft.class_1058;
import net.minecraft.class_286;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_293;
import net.minecraft.class_2960;
import net.minecraft.class_4587;
import net.minecraft.class_757;

/**
 * Utility methods for rendering
 *
 * @author HyCraftHD
 */
public class RenderUtil {
	
	public static final RGBA DARK_CONTAINER_BORDER_COLOR = new RGBA(0x373737FF);
	public static final RGBA MEDIUM_CONTAINER_BORDER_COLOR = new RGBA(0x8B8B8BFF);
	public static final RGBA BRIGHT_CONTAINER_BORDER_COLOR = RGBA.WHITE;
	
	/**
	 * Draws the default container border
	 *
	 * @param poseStack Pose stack
	 * @param x X coordinate
	 * @param y Y coordinate
	 * @param width Width
	 * @param height Height
	 * @param blitOffset zLevel for drawing
	 * @param color The shader color. If using {@link RGBA#WHITE} then the drawing will not be colored
	 */
	public static void drawContainerBorder(class_4587 poseStack, int x, int y, int width, int height, float blitOffset, RGBA color) {
		final class_289 tessellator = class_289.method_1348();
		final class_287 bufferBuilder = tessellator.method_60827(class_293.class_5596.field_27382, class_290.field_1576);
		
		RenderSystem.setShader(class_757::method_34540);
		setShaderColor(color);
		
		RenderSystem.enableBlend();
		RenderSystem.defaultBlendFunc();
		
		addColoredQuad(bufferBuilder, poseStack, x, x + width - 1, y, y + 1, DARK_CONTAINER_BORDER_COLOR, blitOffset);
		addColoredQuad(bufferBuilder, poseStack, x, x + 1, y, y + height - 1, DARK_CONTAINER_BORDER_COLOR, blitOffset);
		
		addColoredQuad(bufferBuilder, poseStack, x + width - 1, x + width, y, y + 1, MEDIUM_CONTAINER_BORDER_COLOR, blitOffset);
		addColoredQuad(bufferBuilder, poseStack, x, x + 1, y + height - 1, y + height, MEDIUM_CONTAINER_BORDER_COLOR, blitOffset);
		
		addColoredQuad(bufferBuilder, poseStack, x + 1, x + width - 1, y + height - 1, y + height, BRIGHT_CONTAINER_BORDER_COLOR, blitOffset);
		addColoredQuad(bufferBuilder, poseStack, x + width - 1, x + width, y + 1, y + height, BRIGHT_CONTAINER_BORDER_COLOR, blitOffset);
		
		class_286.method_43433(bufferBuilder.method_60800());
		
		RenderSystem.disableBlend();
		
		resetShaderColor();
	}
	
	/**
	 * Draws a textured box of any size (smallest size is borderSize * 2 square) based on a fixed size textured box with
	 * continuous borders and filler.
	 *
	 * @param poseStack Pose stack
	 * @param x X coordinate
	 * @param y Y coordinate
	 * @param u U coordinate
	 * @param v V coordinate
	 * @param width Width
	 * @param height Height
	 * @param textureWidth Texture width
	 * @param textureHeight Texture height
	 * @param topBorder Top border
	 * @param bottomBorder Bottom border
	 * @param leftBorder Left border
	 * @param rightBorder Right border
	 * @param blitOffset zLevel for drawing
	 * @param texture Texture location
	 * @param color The shader color. If using {@link RGBA#WHITE} then the image will not be colored
	 */
	public static void drawContinuousTexturedBox(class_4587 poseStack, int x, int y, int u, int v, int width, int height, int textureWidth, int textureHeight, int topBorder, int bottomBorder, int leftBorder, int rightBorder, float blitOffset, class_2960 texture, RGBA color) {
		final int fillerWidth = textureWidth - leftBorder - rightBorder;
		final int fillerHeight = textureHeight - topBorder - bottomBorder;
		final int canvasWidth = width - leftBorder - rightBorder;
		final int canvasHeight = height - topBorder - bottomBorder;
		final int xPasses = canvasWidth / fillerWidth;
		final int remainderWidth = canvasWidth % fillerWidth;
		final int yPasses = canvasHeight / fillerHeight;
		final int remainderHeight = canvasHeight % fillerHeight;
		
		final float uScale = 1f / 256;
		final float vScale = 1f / 256;
		
		final class_289 tessellator = class_289.method_1348();
		final class_287 bufferBuilder = tessellator.method_60827(class_293.class_5596.field_27382, class_290.field_1585);
		
		RenderSystem.setShader(class_757::method_34542);
		RenderSystem.setShaderTexture(0, texture);
		setShaderColor(color);
		
		RenderSystem.enableBlend();
		RenderSystem.defaultBlendFunc();
		
		// Draw Border
		// Top Left
		addTexturedRect(bufferBuilder, poseStack, x, y, u, v, uScale, vScale, leftBorder, topBorder, blitOffset);
		// Top Right
		addTexturedRect(bufferBuilder, poseStack, x + leftBorder + canvasWidth, y, u + leftBorder + fillerWidth, v, uScale, vScale, rightBorder, topBorder, blitOffset);
		// Bottom Left
		addTexturedRect(bufferBuilder, poseStack, x, y + topBorder + canvasHeight, u, v + topBorder + fillerHeight, uScale, vScale, leftBorder, bottomBorder, blitOffset);
		// Bottom Right
		addTexturedRect(bufferBuilder, poseStack, x + leftBorder + canvasWidth, y + topBorder + canvasHeight, u + leftBorder + fillerWidth, v + topBorder + fillerHeight, uScale, vScale, rightBorder, bottomBorder, blitOffset);
		
		for (int index = 0; index < xPasses + (remainderWidth > 0 ? 1 : 0); index++) {
			// Top Border
			addTexturedRect(bufferBuilder, poseStack, x + leftBorder + (index * fillerWidth), y, u + leftBorder, v, uScale, vScale, (index == xPasses ? remainderWidth : fillerWidth), topBorder, blitOffset);
			// Bottom Border
			addTexturedRect(bufferBuilder, poseStack, x + leftBorder + (index * fillerWidth), y + topBorder + canvasHeight, u + leftBorder, v + topBorder + fillerHeight, uScale, vScale, (index == xPasses ? remainderWidth : fillerWidth), bottomBorder, blitOffset);
			
			// Throw in some filler for good measure
			for (int j = 0; j < yPasses + (remainderHeight > 0 ? 1 : 0); j++) {
				addTexturedRect(bufferBuilder, poseStack, x + leftBorder + (index * fillerWidth), y + topBorder + (j * fillerHeight), u + leftBorder, v + topBorder, uScale, vScale, (index == xPasses ? remainderWidth : fillerWidth), (j == yPasses ? remainderHeight : fillerHeight), blitOffset);
			}
		}
		
		// Side Borders
		for (int index = 0; index < yPasses + (remainderHeight > 0 ? 1 : 0); index++) {
			// Left Border
			addTexturedRect(bufferBuilder, poseStack, x, y + topBorder + (index * fillerHeight), u, v + topBorder, uScale, vScale, leftBorder, (index == yPasses ? remainderHeight : fillerHeight), blitOffset);
			// Right Border
			addTexturedRect(bufferBuilder, poseStack, x + leftBorder + canvasWidth, y + topBorder + (index * fillerHeight), u + leftBorder + fillerWidth, v + topBorder, uScale, vScale, rightBorder, (index == yPasses ? remainderHeight : fillerHeight), blitOffset);
		}
		
		class_286.method_43433(bufferBuilder.method_60800());
		
		RenderSystem.disableBlend();
		
		resetShaderColor();
	}
	
	/**
	 * Draws a textured quad.
	 *
	 * @param poseStack Pose stack
	 * @param x X coordinate
	 * @param y Y coordinate
	 * @param width Width
	 * @param height Height
	 * @param uWidth U Width
	 * @param vHeight V Height
	 * @param uOffset U Offset
	 * @param vOffset V Offset
	 * @param textureWidth Texture width
	 * @param textureHeight Texture height
	 * @param blitOffset zLevel for drawing
	 * @param texture Texture location
	 * @param color The shader color. If using {@link RGBA#WHITE} then the image will not be colored
	 */
	public static void drawTexturedQuad(class_4587 poseStack, int x, int y, int width, int height, int uWidth, int vHeight, float uOffset, float vOffset, int textureWidth, int textureHeight, float blitOffset, class_2960 texture, RGBA color) {
		drawTexturedQuad(poseStack, x, x + width, y, y + height, uOffset / textureWidth, (uOffset + uWidth) / textureWidth, vOffset / textureHeight, (vOffset + vHeight) / textureHeight, blitOffset, texture, color);
	}
	
	/**
	 * Draws a textured quad.
	 *
	 * @param poseStack Pose stack
	 * @param x X coordinate
	 * @param y Y coordinate
	 * @param width Width
	 * @param height Height
	 * @param blitOffset zLevel for drawing
	 * @param sprite Texture sprite from texture atlas
	 * @param color The shader color. If using {@link RGBA#WHITE} then the image will not be colored
	 */
	public static void drawTexturedQuad(class_4587 poseStack, int x, int y, int width, int height, float blitOffset, class_1058 sprite, RGBA color) {
		drawTexturedQuad(poseStack, x, x + width, y, y + height, sprite.method_4594(), sprite.method_4577(), sprite.method_4593(), sprite.method_4575(), blitOffset, sprite.method_45852(), color);
	}
	
	/**
	 * Draws a textured quad.
	 *
	 * @param poseStack Pose stack
	 * @param x1 X1 coordinate
	 * @param x2 X2 coordinate
	 * @param y1 Y1 coordinate
	 * @param y2 Y2 coordinate
	 * @param u1 U1 coordinate
	 * @param u2 U2 coordinate
	 * @param v1 V1 coordinate
	 * @param v2 V2 coordinate
	 * @param blitOffset zLevel for drawing
	 * @param texture Texture location
	 * @param color The shader color. If using {@link RGBA#WHITE} then the image will not be colored
	 */
	public static void drawTexturedQuad(class_4587 poseStack, int x1, int x2, int y1, int y2, float u1, float u2, float v1, float v2, float blitOffset, class_2960 texture, RGBA color) {
		final class_289 tessellator = class_289.method_1348();
		final class_287 bufferBuilder = tessellator.method_60827(class_293.class_5596.field_27382, class_290.field_1585);
		
		RenderSystem.setShader(class_757::method_34542);
		RenderSystem.setShaderTexture(0, texture);
		setShaderColor(color);
		
		RenderSystem.enableBlend();
		RenderSystem.defaultBlendFunc();
		
		addTexturedQuad(bufferBuilder, poseStack, x1, x2, y1, y2, u1, u2, v1, v2, blitOffset);
		
		class_286.method_43433(bufferBuilder.method_60800());
		
		RenderSystem.disableBlend();
		
		resetShaderColor();
	}
	
	/**
	 * Adds a textured rectangle to the buffer builder. The vertex format must be {@link class_290#field_1585}
	 * and the draw format must be {@link class_293.class_5596#field_27382}
	 *
	 * @param bufferBuilder Buffer builder
	 * @param poseStack Pose stack
	 * @param x X coordinate
	 * @param y Y coordinate
	 * @param u U coordinate
	 * @param v V coordinate
	 * @param uScale U scale
	 * @param vScale V scale
	 * @param width Width
	 * @param height Height
	 * @param blitOffset zLevel for drawing
	 */
	public static void addTexturedRect(class_287 bufferBuilder, class_4587 poseStack, int x, int y, int u, int v, float uScale, float vScale, int width, int height, float blitOffset) {
		final Matrix4f matrix = poseStack.method_23760().method_23761();
		
		bufferBuilder.method_22918(matrix, x, y + height, blitOffset).method_22913(u * uScale, ((v + height) * vScale));
		bufferBuilder.method_22918(matrix, x + width, y + height, blitOffset).method_22913((u + width) * uScale, ((v + height) * vScale));
		bufferBuilder.method_22918(matrix, x + width, y, blitOffset).method_22913((u + width) * uScale, (v * vScale));
		bufferBuilder.method_22918(matrix, x, y, blitOffset).method_22913(u * uScale, (v * vScale));
	}
	
	/**
	 * Adds a textured quad to the buffer builder. The vertex format must be {@link class_290#field_1585} and
	 * the draw format must be {@link class_293.class_5596#field_27382}
	 *
	 * @param bufferBuilder Buffer builder
	 * @param poseStack Pose stack
	 * @param x1 X1 coordinate
	 * @param x2 X2 coordinate
	 * @param y1 Y1 coordinate
	 * @param y2 Y2 coordinate
	 * @param u1 U1 coordinate
	 * @param u2 U2 coordinate
	 * @param v1 V1 coordinate
	 * @param v2 V2 coordinate
	 * @param blitOffset zLevel for drawing
	 */
	public static void addTexturedQuad(class_287 bufferBuilder, class_4587 poseStack, int x1, int x2, int y1, int y2, float u1, float u2, float v1, float v2, float blitOffset) {
		final Matrix4f matrix = poseStack.method_23760().method_23761();
		
		bufferBuilder.method_22918(matrix, x1, y2, blitOffset).method_22913(u1, v2);
		bufferBuilder.method_22918(matrix, x2, y2, blitOffset).method_22913(u2, v2);
		bufferBuilder.method_22918(matrix, x2, y1, blitOffset).method_22913(u2, v1);
		bufferBuilder.method_22918(matrix, x1, y1, blitOffset).method_22913(u1, v1);
	}
	
	/**
	 * Adds a quad to the buffer builder. The vertex format must be {@link class_290#field_1576} and the draw
	 * format must be {@link class_293.class_5596#field_27382}
	 *
	 * @param bufferBuilder Buffer builder
	 * @param poseStack Pose stack
	 * @param x1 X1 coordinate
	 * @param x2 X2 coordinate
	 * @param y1 Y1 coordinate
	 * @param y2 Y2 coordinate
	 * @param color Color of the vertices
	 * @param blitOffset zLevel for drawing
	 */
	public static void addColoredQuad(class_287 bufferBuilder, class_4587 poseStack, int x1, int x2, int y1, int y2, RGBA color, float blitOffset) {
		final Matrix4f matrix = poseStack.method_23760().method_23761();
		
		bufferBuilder.method_22918(matrix, x1, y2, blitOffset).method_1336(color.getRed(), color.getGreen(), color.getBlue(), color.getAlpha());
		bufferBuilder.method_22918(matrix, x2, y2, blitOffset).method_1336(color.getRed(), color.getGreen(), color.getBlue(), color.getAlpha());
		bufferBuilder.method_22918(matrix, x2, y1, blitOffset).method_1336(color.getRed(), color.getGreen(), color.getBlue(), color.getAlpha());
		bufferBuilder.method_22918(matrix, x1, y1, blitOffset).method_1336(color.getRed(), color.getGreen(), color.getBlue(), color.getAlpha());
	}
	
	/**
	 * Adds a quad to the buffer builder. The vertex format must be {@link class_290#field_1592} and the draw format
	 * must be {@link class_293.class_5596#field_27382}
	 *
	 * @param bufferBuilder Buffer builder
	 * @param poseStack Pose stack
	 * @param x1 X1 coordinate
	 * @param x2 X2 coordinate
	 * @param y1 Y1 coordinate
	 * @param y2 Y2 coordinate
	 * @param blitOffset zLevel for drawing
	 */
	public static void addQuad(class_287 bufferBuilder, class_4587 poseStack, int x1, int x2, int y1, int y2, float blitOffset) {
		final Matrix4f matrix = poseStack.method_23760().method_23761();
		
		bufferBuilder.method_22918(matrix, x1, y2, blitOffset);
		bufferBuilder.method_22918(matrix, x2, y2, blitOffset);
		bufferBuilder.method_22918(matrix, x2, y1, blitOffset);
		bufferBuilder.method_22918(matrix, x1, y1, blitOffset);
	}
	
	/**
	 * Sets the shader color from {@link RGBA} type
	 *
	 * @param rgba Color
	 */
	public static void setShaderColor(RGBA rgba) {
		RenderSystem.setShaderColor(rgba.getRedComponent(), rgba.getGreenComponent(), rgba.getBlueComponent(), rgba.getAlphaComponent());
	}
	
	/**
	 * Sets the shader color to {@link RGBA#WHITE}
	 */
	public static void resetShaderColor() {
		setShaderColor(RGBA.WHITE);
	}
	
}
